<?php

namespace App\Http\Controllers;

use App\Exceptions\InvalidRequestException;
use App\Models\Roles;
use App\Models\Sucursal;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Route;
use App\Http\Controllers\exception;
use App\Http\Controllers\Permission;

use Spatie\Permission\Traits\HasRoles;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Authh;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Models\Recompensa;
use App\Models\RecompensaSucursal;
use App\Models\RecompensaCategoria;
use App\Models\RecompensaPromocional;

class RecompensasController extends Controller
{
    use HasRoles;

    public function __construct()
    {
        $this->middleware('permission:listar_recompensas', ['only' => ['index']]);
        $this->middleware('permission:crear_recompensas', ['only' => ['create','store']]);
        $this->middleware('permission:editar_recompensas', ['only' => ['edit','update']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $usuarioAuth = Auth::user();

        $recompensas = Recompensa::select('id','status','idrecompensa_categoria','nombre');
        // Sin permiso solo ve la sucursal que tiene asignada
        if (!$usuarioAuth->can('ver_todas_sucursales')) {
            $recompensas->whereHas('Sucursal', function ($query) use ($usuarioAuth) {
                $query->where('idsucursal', $usuarioAuth->idsucursal);
            });
        }
        $recompensas = $recompensas->get();

        return view('Recompensas.recompensas_list', compact('recompensas'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $usuarioAuth = Auth::user();

        $sucursales = Sucursal::where('status', true)->orderBy('nombre');
        if (!$usuarioAuth->can('ver_todas_sucursales')) {
            $sucursales->where('id', $usuarioAuth->sucursal->id);
        }
        $sucursales = $sucursales->get();

        $categorias = RecompensaCategoria::where('status', true)->get();

        return view('Recompensas.recompensas_create', compact('sucursales','categorias'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            if(Recompensa::whereRaw("nombre ilike '$request->nombre'")->exists()){
                throw new \Exception("El nombre ya existe", 1);
            }

            DB::beginTransaction();

            $oRecompensa = new Recompensa();
            $oRecompensa->status = $request->status == 'true';
            $oRecompensa->idrecompensa_categoria = $request->id_categoria;
            $oRecompensa->nombre = $request->nombre;
            $oRecompensa->inicio_fecha = $request->inicio_fecha;
            $oRecompensa->inicio_hora = $request->inicio_hora;
            $oRecompensa->fin_fecha = $request->fin_fecha;
            $oRecompensa->fin_hora = $request->fin_hora;
            $oRecompensa->save();

            foreach($request->id_sucursal as $idSucursal){
                RecompensaSucursal::insert([
                    'idrecompensa' => $oRecompensa->id,
                    'idsucursal' => $idSucursal
                ]);
            }
            
            $promocional_counter = 1;

            foreach($request->articulo_promocional as $promocional){

                $img = getB64Image($promocional['imagen']);
                $img_extension = getB64Extension($promocional['imagen']);
                if(!in_array($img_extension, ['jpg', 'jpeg', 'png', 'gif'])){
                    throw new \Exception("Extensión de imagen inválida", 1);
                }
                $img_name = 'img_promocional_'. time() . '_' . $promocional_counter++ . '.' . $img_extension;
                Storage::disk('recompensapromocional')->put($oRecompensa->id . '/' .$img_name, $img);

                $oRecompensaPromocional = new RecompensaPromocional();
                $oRecompensaPromocional->idrecompensa = $oRecompensa->id;
                $oRecompensaPromocional->nombre = $promocional['nombre'];
                $oRecompensaPromocional->codigo = $promocional['codigo'];
                $oRecompensaPromocional->valor = $promocional['valor'];
                $oRecompensaPromocional->descripcion = $promocional['descripcion'];
                $oRecompensaPromocional->imagen = $img_name;
                $oRecompensaPromocional->save();

                //Save with full path
                $img_name_DB = env('APP_FOLIATTI_IMG_PATH') . 'recompensas/promocional/' . $oRecompensaPromocional->idrecompensa . '/' .   $img_name;
                $oRecompensaPromocional->imagen_fullpath = $img_name_DB;            
                $oRecompensaPromocional->save();
            }

            DB::commit();
        } catch (\Exception $e) {
            throw new InvalidRequestException($e->getMessage().$e->getLine());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function show(User $user)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $usuarioAuth = Auth::user();

        $sucursales = Sucursal::where('status', true)->orderBy('nombre')->get();

        $idSucursalUsuario = null;
        if (!$usuarioAuth->can('ver_todas_sucursales')) {
            $idSucursalUsuario = $usuarioAuth->sucursal->id;
        }

        $categorias = RecompensaCategoria::where('status', true)->get();

        $recompensa = Recompensa::findOrFail($id);

        $aIdSucursalRecompensa = $recompensa->sucursal()->pluck('id')->toArray();

        $aArticuloPromocional = $recompensa->promocional()->get()->toArray();

        return view('Recompensas.recompensas_create', compact('sucursales', 'idSucursalUsuario', 'categorias', 'recompensa', 'aIdSucursalRecompensa', 'aArticuloPromocional'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            if(Recompensa::whereRaw("nombre ilike '$request->nombre'")->where('id','<>',$id)->exists()){
                throw new \Exception("El nombre ya existe", 1);
            }

            DB::beginTransaction();

            $oRecompensa = Recompensa::findOrFail($id);
            $oRecompensa->status = $request->status == 'true';
            $oRecompensa->idrecompensa_categoria = $request->id_categoria;
            $oRecompensa->nombre = $request->nombre;
            $oRecompensa->inicio_fecha = $request->inicio_fecha;
            $oRecompensa->inicio_hora = $request->inicio_hora;
            $oRecompensa->fin_fecha = $request->fin_fecha;
            $oRecompensa->fin_hora = $request->fin_hora;
            $oRecompensa->save();

            if (Auth::user()->can('ver_todas_sucursales')) {
                $oRecompensa->recompensaSucursal()->delete();
                foreach($request->id_sucursal as $idSucursal){
                    RecompensaSucursal::insert([
                        'idrecompensa' => $id,
                        'idsucursal' => $idSucursal
                    ]);
                }
            }

            $aIdPromocionalNoDelete = [];
            $promocional_counter = 1;
            
            foreach($request->articulo_promocional as $promocional){
                if(isset($promocional['id'])){
                    $aIdPromocionalNoDelete[] = $promocional['id'];
                }else{
                    $img = getB64Image($promocional['imagen']);
                    $img_extension = getB64Extension($promocional['imagen']);
                    if(!in_array($img_extension, ['jpg', 'jpeg', 'png', 'gif'])){
                        throw new \Exception("Extensión de imagen inválida", 1);
                    }
                    $img_name = 'img_promocional_'. time() . '_' . $promocional_counter++ . '.' . $img_extension;
                    Storage::disk('recompensapromocional')->put($oRecompensa->id . '/' .$img_name, $img);

                    $oRecompensaPromocional = new RecompensaPromocional();
                    $oRecompensaPromocional->idrecompensa = $id;
                    $oRecompensaPromocional->nombre = $promocional['nombre'];
                    $oRecompensaPromocional->codigo = $promocional['codigo'];
                    $oRecompensaPromocional->valor = $promocional['valor'];
                    $oRecompensaPromocional->descripcion = $promocional['descripcion'];
                    $oRecompensaPromocional->imagen = $img_name;
//                    $oRecompensaPromocional->save();

                    //Save with full path
                    $img_name_DB = env('APP_FOLIATTI_IMG_PATH') . 'recompensas/promocional/' . $oRecompensaPromocional->idrecompensa . '/' .   $img_name;
                    $oRecompensaPromocional->imagen_fullpath = $img_name_DB;            
                    $oRecompensaPromocional->save();


                    $aIdPromocionalNoDelete[] = $oRecompensaPromocional->id;
                }
            }

            // borramos solo los registros de recompensa_promocional que no regresaron
            $oRecompensa->promocional()->whereNotIn('id', $aIdPromocionalNoDelete)->delete();

            DB::commit();
        } catch (\Exception $e) {
            throw new InvalidRequestException($e->getMessage().$e->getLine());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $oRecompensa = Recompensa::findOrFail($id);
        $oRecompensa->delete();
    }

}
